/*
 * Copyright (c) Doug Palmer <doug@charvolant.org> 2005
 *
 * See LICENSE for licensing details.
 * 
 * $Id$
 */

package org.charvolant.sudoku;

import java.util.*;
import org.charvolant.sudoku.gui.*;

/**
 * A board of squares.
 * 
 * @author doug
 *
 */
public class Board extends Model {
  /** The size of one side of the board */
  private int size;
  /** The size of one size of a square */
  private int sqsize;
  /** The cells */
  private Cell[][] cells;
  /** The squares */
  private Square[] squares;
  /** The rows */
  private Strip[] rows;
  /** The columns */
  private Strip[] columns;
  /** The context that this board uses */
  private Context context;
  
  /**
   * Constuct a board of a specific size.
   * <p>
   * Sizes need to be a sqaure, so that the board can be
   * correctly tiled with squares. 4, 9, 16, 25 are all acceptable
   * values.
   * 
   * @param size The size of a side of the board in whole sqaures.
   */
  public Board(int size) {
    if (size != 4 && size != 9 && size != 16 && size != 25)
      throw new IllegalArgumentException("Invalid board size of " + size);
    this.size = size;
    for (this.sqsize = 1; this.sqsize * this.sqsize < this.size; this.sqsize++);
    this.reset();
  }
  
  /**
   * Reset the board to be empty.
   */
  public void reset() {
    int size = this.getSize();
    int i, j;
    
    this.context = new Context();
    this.cells = new Cell[size][size];
    for (i = 0; i < size; i++)
      for (j = 0; j < size; j++)
        this.cells[i][j] = new Cell(this);
    this.squares = new Square[size];
    for (i = 0; i < size; i++) {
      this.squares[i] = new Square(this);
      for (j = 0; j < size; j++)
        this.cells[this.getSquareY(i) * this.sqsize + this.getSquareY(j)][this.getSquareX(i) * this.sqsize + this.getSquareX(j)].addShape(this.squares[i]);
    }
    this.rows = new Strip[size];
    for (i = 0; i < size; i++) {
      this.rows[i] = new Strip(this);
      for (j = 0; j < size; j++)
        this.cells[i][j].addShape(this.rows[i]);
    }
    this.columns = new Strip[size];
    for (i = 0; i < size; i++) {
      this.columns[i] = new Strip(this);
      for (j = 0; j < size; j++)
        this.cells[j][i].addShape(this.columns[i]);
    }
  }

  /**
   * Get the size of the board.
   * <p>
   * This is the size of one side of the board.
   * 
   * @return The side size
   */
  public int getSize() {
    return this.size;
  }

  /**
   * Get the square size of the board.
   * <p>
   * This is the size of one side of a square.
   * This works either for laying out possibilities in a single
   * cell or for the larger squares.
   * 
   * @return The square side size
   */
  public int getSquareSize() {
    return this.sqsize;
  }
   
  /**
   * Get the context.
   *
   * @return Returns the context.
   */
  public Context getContext() {
    return this.context;
  }
  
  /**
   * Get the Y position of a square.
   * <p>
   * Squares are numbered across rows, eg.
   * <p>
   * <pre>
   * 0 1 2
   * 3 4 5
   * 6 7 8
   * </pre>
   * 
   * @param sn The square number
   * 
   * @return The y position
   */
  public int getSquareY(int sn) {
    return sn / this.sqsize;
  }
  
  /**
   * Get the X position of a square.
   * <p>
   * Squares are numbered across rows, eg.
   * <p>
   * <pre>
   * 0 1 2
   * 3 4 5
   * 6 7 8
   * </pre>
   * 
   * @param sn The square number
   * 
   * @return The X position
   */
  public int getSquareX(int sn) {
    return sn % this.sqsize;
  }
  
  /**
   * Search for deductions in the shapes that we have.
   */
  public void search() {
    int i;
    
    for (i = 0; i < this.squares.length; i++)
      this.squares[i].search();
    for (i = 0; i < this.rows.length; i++)
      this.rows[i].search();
    for (i = 0; i < this.columns.length; i++)
      this.columns[i].search();
  }
  
  /**
   * Make a step in solving the puzzle.
   * <p>
   * First find all the singletons, then fix
   * them.
   *
   */
  public void step() {
    Collection<Cell> singletons = this.getSingletons();
    
    for (Cell cell: singletons)
      cell.fix(cell.getSingleton());
    this.search();
  }
  
  /**
   * Get the singleton cells.
   * 
   * @return A collection of singleton cells
   */
  public Collection<Cell> getSingletons() {
    Collection<Cell> singletons = new ArrayList<Cell>();
    int size = this.getSize();
    
    for (int i = 0; i < size; i++)
      for (int j = 0; j < size; j++)
        if (this.cells[i][j].isSingleton())
          singletons.add(this.cells[i][j]);
    return singletons;
  }
  
  /**
   * Get all the cells.
   * 
   * @return The 2-D array of cells
   */
  public Cell[][] getCells() {
    return this.cells;
  }
  
  /**
   * Run with a new board.
   * <p>
   * The first parameter is the board size, or 9 by default.
   */
  public static void main(String[] args) {
    try {
      int size = 9;
      if (args.length > 0)
        size = Integer.parseInt(args[0]);
      Board board = new Board(size);
      GamePanel panel = new GamePanel(board);
      panel.openFrame();
    } catch (Exception ex) {
      ex.printStackTrace(System.err);
    }
  }
}
